"""
HB_SmoothEdge V1.2

Last Modified: Oct/10/2018
Works with CINEMA 4D R16.050 and up.
Copyright: Holger Biebrach, www.c4dstuff.com

Name-US: HB_SmoothEdge
Description-US: Converts current Edgeselection to a Parametric Spline-Deformer Setup

Usage:
Make an Edgeselektion and run the Script. In the Dialog you have following Settings:

- CurvePoints: Sets the amount of Points for the modifying Spline
- Radius Factor: Sets the default Influence Radius in relation to the Splinelength (0.2 by default)
- Precise Spine: If on the Modifier Spline will be exactly on the Edgeselection. Carefull: this will allready deform the Mesh a bit as
the Interpolation will be changed. If this option is off the mesh will not change initially but the Modifierspline will not be precicly on the Edges.
- Restrict to Pointselection: If you have Points selected this option will restrict the Deformations to the selected Points.
- Max Handles: this will place a ModifierPoint on each selected Edge-Point. Also it will automatically choose the Magnet-Tool

Info: You can use the HB_BrushradiusIncrease/decrease scripts for changing the influence-Radius

When you are done with your Modifications you need to run the HB_SmoothEdgeConvert Script to convert everything to a Mesh again.

Video Tutorial:
https://youtu.be/Ki6I9sRdeUU?t=28m16s

https://youtu.be/UlY-X5O137M?t=249

ChangeLog:

Jun/11/2015 V1.0
- Release Version

Jan/1/2016 V1.1
- minor changes
- support for Retopo Workflow

Oct/10/2018 V1.2
- Python rewrite
- Removed option for Spline Interpolation
- New Highres Icon

"""
import c4d
from c4d import gui, utils

class SettingsDialog(gui.GeDialog):
    annuler = True
    handles = 3
    PreciseSpline = False
    RestrictPoints = False
    RadiusFactor = 0
    MaxHandles = False


    def CreateLayout(self):
        self.SetTitle("HB_SmoothEdge")
        self.GroupBegin(10, 2, 2)
        self.AddStaticText(11, c4d.BFH_SCALEFIT, 0, 0, "Handles :", 0)
        self.AddEditNumberArrows(12, c4d.BFH_SCALEFIT, 0, 0)
        self.AddStaticText(17, c4d.BFH_SCALEFIT, 0, 0, "Radius Factor :", 0)
        self.AddEditNumberArrows(18, c4d.BFH_SCALEFIT, 0, 0)
        self.AddCheckbox(13, c4d.BFH_SCALEFIT, 0, 0, "Precise Spline")
        self.AddCheckbox(14, c4d.BFH_SCALEFIT, 0, 0, "Restrict to Ponintselection")
        self.AddCheckbox(141, c4d.BFH_SCALEFIT, 0, 0, "Max Handles")
        self.GroupEnd()
        self.AddSeparatorV(15)
        self.AddButton(19, c4d.BFH_SCALEFIT, initw = 100, inith = 20, name = "Ok")
        self.AddButton(191, c4d.BFH_SCALEFIT, initw = 100, inith = 20, name = "Cancel")
        self.AddStaticText(16, c4d.BFH_SCALEFIT, 0, 0, "www.c4dstuff.com", 1)
        return True

    def InitValues(self) :
        self.SetLong(12, 3, 3, 1000, 1)
        self.SetBool(13, False)
        self.SetBool(16, False)
        self.SetFloat(18, 0.2)
        return True

    def Command(self, id, msg):
        if (id == 13):
            self.Enable(14, self.GetBool(13))
        if (id == 19):
            self.annuler = False
            self.handles = self.GetLong(12)
            self.PreciseSpline = self.GetBool(13)
            self.RestrictPoints = self.GetBool(14)
            self.RadiusFactor = self.GetReal(18)
            self.MaxHandles = self.GetBool(141)
            self.Close()
        if (id == 191):
            self.annuler = True
            self.Close()
        return True


def InverseDelete(obj):
    BCinverse = c4d.BaseContainer()
    utils.SendModelingCommand(command=c4d.MCOMMAND_SELECTINVERSE, list=[obj], mode=c4d.MODIFY_POINTSELECTION, bc=BCinverse, doc=doc)

    BCdelete = c4d.BaseContainer()
    utils.SendModelingCommand(command=c4d.MCOMMAND_DELETE, list=[obj], mode=c4d.MODIFY_POINTSELECTION, bc=BCdelete, doc=doc)


def DeselectAll(obj):
    BCdeselect = c4d.BaseContainer()
    utils.SendModelingCommand(command=c4d.MCOMMAND_DESELECTALL, list=[obj], mode=c4d.MODIFY_POINTSELECTION, bc=BCdeselect, doc=doc)


def selectPoints(selobj):

    segPrecount = selobj.GetPointCount()
    segSelect   = selobj.GetPointS()

                                                                                            #only two or three handle
    if dialog.handles== 2 or dialog.handles== 3:

        segSelectCopy=c4d.BaseSelect()


        segSelectCopy.Select(segPrecount-1)
        segSelectCopy.Select(0)




        if (dialog.handles==3):                                                             #three handles

            segSelectCopy.Select(segPrecount/2)

        segSelectCopy.CopyTo(segSelect)

    else:                                                                                 #more than 3 handles
        number=segPrecount/dialog.handles
        segSelectCopy=c4d.BaseSelect()





        i = 0

        while(i<=segPrecount):

            segSelectCopy.Select(i)
            i = i+number
        segSelectCopy.Select(0)
        segSelectCopy.CopyTo(segSelect)


        if(selobj[c4d.SPLINEOBJECT_CLOSED]==False):
            segSelect.Select(segPrecount-1)


        selobj.Message(c4d.MSG_UPDATE)


######################################  MAIN SCRIPT ###################

def SmoothEdge():
    global dialog
    
    obj=doc.GetActiveObject()
    if not obj:
        gui.MessageDialog("Select one Polygonobject")
        return
    
    dialog = SettingsDialog()
    dialog.Open(c4d.DLG_TYPE_MODAL)
    if (dialog.annuler is True): return




    doc.StartUndo()


    if (c4d.IsCommandChecked(12139)) and dialog.RestrictPoints==False: #PointsMode

        ConvertSettings = c4d.BaseContainer()

        ConvertSettings[c4d.MDATA_CONVERTSELECTION_LEFT]=0
        ConvertSettings[c4d.MDATA_CONVERTSELECTION_RIGHT]=1

        c4d.utils.SendModelingCommand(command = c4d.MCOMMAND_CONVERTSELECTION,
                                    list = [obj],
                                    mode = c4d.MODELINGCOMMANDMODE_EDGESELECTION,
                                    bc = ConvertSettings,
                                    doc = doc)
                                                                                                    #EDGE TO SPLINE


    settings = c4d.BaseContainer()
    utils.SendModelingCommand(command = c4d.MCOMMAND_EDGE_TO_SPLINE,
                                list = [obj],
                                mode = c4d.MODELINGCOMMANDMODE_EDGESELECTION,
                                bc = settings,
                                doc = doc,
                                flags = c4d.MODELINGCOMMANDFLAGS_CREATEUNDO)

    orgSpline=obj.GetDown()
    obj.DelBit(c4d.BIT_ACTIVE)
    orgSpline.SetBit(c4d.BIT_ACTIVE)
    segmentcount=orgSpline.GetSegmentCount()



                                                                                                    #Check pointcount (befor/after optimize) to see if severall loops


    pointcount= orgSpline.GetPointCount()


                                                                                                    #Optimize
    OptCommand= c4d.MCOMMAND_OPTIMIZE
    mode = c4d.MODIFY_ALL
    options = c4d.BaseContainer()
    options[c4d.MDATA_OPTIMIZE_TOLERANCE]=0.001
    options[c4d.MDATA_OPTIMIZE_POINTS] = True
    options[c4d.MDATA_OPTIMIZE_POLYGONS] = True
    utils.SendModelingCommand(command=OptCommand, list=[orgSpline], mode=mode, bc=options, doc=doc)

    newpointcount= orgSpline.GetPointCount()

    if pointcount != newpointcount:

        orgSpline[c4d.SPLINEOBJECT_CLOSED]=True


                                                                                                   #Calculate Radius of Splinedeformer-Fallof

    rs = orgSpline.GetRealSpline()
    sld=c4d.utils.SplineLengthData(rs,0)

    sld.Init(rs,0)
    length = sld.GetLength()
    radius = dialog.RadiusFactor*length


                                                                                                   #Spline has one segment

    if segmentcount ==1:
        if dialog.PreciseSpline == False:
            selectPoints(orgSpline)

            if dialog.MaxHandles == False:
                InverseDelete(orgSpline)
            else:
                DeselectAll(orgSpline)



                                                                                                   #Spline has MORE than one segment
    else:
        if dialog.PreciseSpline == False:
            BCexplode = c4d.BaseContainer()
            utils.SendModelingCommand(command=c4d.MCOMMAND_EXPLODESEGMENTS, list=[orgSpline], mode=c4d.MODIFY_POINTSELECTION, bc=BCexplode, doc=doc)
            parentspline = doc.GetActiveObject()
            child=parentspline.GetDown()
            while child:
                child.SetBit(c4d.BIT_ACTIVE)
                child=child.GetNext()
            parentspline.DelBit(c4d.BIT_ACTIVE)
            ObjList = doc.GetActiveObjects(True)
            for Obj in ObjList:
                selectPoints(Obj)


            c4d.CallCommand(16768) # connect&delete
            newspline=doc.GetActiveObject()
            if dialog.MaxHandles == False:                                      #MaxHandles = FALSE
                InverseDelete(newspline)


            parentspline.SetBit(c4d.BIT_ACTIVE)
            newspline.DelBit(c4d.BIT_ACTIVE)
            childspline=parentspline.GetDown()
            childspline.Remove()
            parentspline.Remove()
            childspline.InsertUnder(obj)
            childspline.SetBit(c4d.BIT_ACTIVE)
            newspline=doc.GetActiveObject()


                                                                                        #TARGETSPLINE

    defaultspline=doc.GetActiveObject()
    rootobj=defaultspline.GetUp()
    defaultspline.SetName("HB_Orginal_Spline")
    targetspline=defaultspline.GetClone(0)
    targetspline.InsertUnder(rootobj)
    targetspline.SetName("HB_Modify_Spline")
    pos=defaultspline.GetMg()
    targetspline.SetMg(pos)
    defaultspline.DelBit(c4d.BIT_ACTIVE)



    if dialog.PreciseSpline == True and segmentcount==1:                                    #Precise is on and one Segment
        selectPoints(targetspline)
        if dialog.MaxHandles == False:                                                    #MaxHandles = FALSE
            InverseDelete(targetspline)
        else:

            DeselectAll(targetspline)



                                                                                        #Precise on and more Segments
    if dialog.PreciseSpline == True and segmentcount!=1:
        BCexplode = c4d.BaseContainer()
        utils.SendModelingCommand(command=c4d.MCOMMAND_EXPLODESEGMENTS, list=[targetspline], mode=c4d.MODIFY_POINTSELECTION, bc=BCexplode, doc=doc)
        parentspline = doc.GetActiveObject()
        child=parentspline.GetDown()

        while child:
            child.SetBit(c4d.BIT_ACTIVE)
            child=child.GetNext()


        parentspline.DelBit(c4d.BIT_ACTIVE)
        ObjList = doc.GetActiveObjects(True)
        for Obj in ObjList:
            selectPoints(Obj)



        ExplodeSplines=doc.GetActiveObjects(0)



        targetspline = utils.SendModelingCommand(command = c4d.MCOMMAND_JOIN,
                            list = [parentspline],
                            mode = c4d.MODELINGCOMMANDMODE_ALL,
                            doc = doc)[0]



        targetspline.InsertUnder(obj)

        targetspline.Remove()
        parentspline.Remove()
        targetspline.InsertUnder(obj)


        if dialog.MaxHandles == False:                                                    #MaxHandles = FALSE
            InverseDelete(targetspline)

        else:
           DeselectAll(targetspline)



                                                                                            # SPLINE DEFORMER SETUP


    splineDeformer = c4d.BaseObject(c4d.Osplinedeformer)
    splineDeformer.InsertUnder(obj)
    splineDeformer[c4d.SPLINEDEFORMEROBJECT_ORIGINAL_SPLINE]=defaultspline
    splineDeformer[c4d.SPLINEDEFORMEROBJECT_MODIFY_SPLINE]=targetspline

    if dialog.PreciseSpline == True:
        splineDeformer[c4d.SPLINEDEFORMEROBJECT_USE_LENGTH]=True
    if dialog.MaxHandles == True:
         splineDeformer[c4d.SPLINEDEFORMEROBJECT_USE_LENGTH]=False
    if targetspline[c4d.SPLINEOBJECT_CLOSED]==True:
         splineDeformer[c4d.SPLINEDEFORMEROBJECT_USE_LENGTH]=False
    splineDeformer[c4d.SPLINEDEFORMEROBJECT_RADIUS]=radius
    splineDeformer.DelBit(c4d.BIT_ACTIVE)
    splineDeformer.SetName("HB_Smoothedge_Deformer")
    splineDeformer.Message(c4d.MSG_DESCRIPTION_VALIDATE)
    #splineDeformer

    if pointcount%dialog.handles != 0:
        print pointcount%dialog.handles
        splineDeformer[c4d.SPLINEDEFORMEROBJECT_USE_LENGTH]=True

                                                                                            #SPLINES SETUP
    if dialog.PreciseSpline == True:
        defaultspline[c4d.SPLINEOBJECT_TYPE]=0
    else:
        defaultspline[c4d.SPLINEOBJECT_TYPE]=4

    if dialog.MaxHandles == True:
        targetspline[c4d.SPLINEOBJECT_TYPE]=0
    else:
        targetspline[c4d.SPLINEOBJECT_TYPE]=4

    defaultspline[c4d.ID_BASEOBJECT_VISIBILITY_EDITOR]=1
    targetspline[c4d.SPLINEOBJECT_INTERPOLATION]=3


                                                                                        #Resrict to Pointselection
    if dialog.RestrictPoints == True:
        obj.SetBit(c4d.BIT_ACTIVE)
        doc.SetMode(c4d.Mpoints)
        targetspline.DelBit(c4d.BIT_ACTIVE)


        c4d.CallCommand(12552) # Set Selection
        seltag=doc.GetActiveTag()
        seltag[c4d.ID_BASELIST_NAME]="HB_Smoothedge_Restriction"
        restag =  c4d.BaseTag(5683)
        splineDeformer.InsertTag(restag)
        restag[c4d.RESTRICTIONTAG_NAME_01]="HB_Smoothedge_Restriction"
        splineDeformer[c4d.SPLINEDEFORMEROBJECT_RADIUS]=10000000000000000


                                                                                            #Final Settings
    obj.DelBit(c4d.BIT_ACTIVE)
    targetspline.SetBit(c4d.BIT_ACTIVE)
    c4d.CallCommand(200000088) # Move
    doc.SetMode(c4d.Mpoints)
    if dialog.handles == 3 and segmentcount==1 and dialog.MaxHandles == False:
        pointcount=targetspline.GetPointCount()
        Select   = targetspline.GetPointS()
        SelectCopy=c4d.BaseSelect()
        SelectCopy.Select(pointcount/2)
        SelectCopy.CopyTo(Select)

    #MagnetTool
    if dialog.MaxHandles == True: c4d.CallCommand(1016185) # Magnet

    doc.GetActiveBaseDraw()[c4d.BASEDRAW_DISPLAYFILTER_SPLINE]=True



    doc.EndUndo()
    c4d.EventAdd()

if __name__=='__main__':
    SmoothEdge()